//	TorusGamesOptionsChoiceController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesOptionsChoiceController.h"
#import "GeometryGamesUtilities-iOS.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"


struct
{
	bool	itsHumanVsComputer;
	Char16	*itsName,
			*itsImageName;
} gPlayModeNames[] =
{
	{false,	u"Human vs. Human",		u"Table Images/Options/Players/Human vs Human"		},
	{true,	u"Human vs. Computer",	u"Table Images/Options/Players/Human vs Computer"	}
};

typedef struct
{
	TopologyType	itsTopology;
	Char16			*itsName,
					*itsImageName;
} TopologyName;
TopologyName
	gTopology2DNames[] =
	{
		{Topology2DTorus,		u"Torus",				u"Table Images/Options/Topology/2D Torus"				},
		{Topology2DKlein,		u"Klein Bottle",		u"Table Images/Options/Topology/2D Klein Bottle"		}
	},
	gTopology3DNames[] =
	{
		{Topology3DTorus,		u"3-Torus",				u"Table Images/Options/Topology/3D Three-Torus"			},
		{Topology3DKlein,		u"Klein Space",			u"Table Images/Options/Topology/3D Klein Space"			},
		{Topology3DQuarterTurn, u"Quarter-Turn Space",	u"Table Images/Options/Topology/3D Quarter-Turn Space"	},
		{Topology3DHalfTurn,	u"Half-Turn Space",		u"Table Images/Options/Topology/3D Half-Turn Space"		}
	};

struct
{
	unsigned int	itsDifficultyLevel;	//	corresponds to ModelData's itsDifficultyLevel
	Char16			*itsName,
					*itsImageName;
} gDifficultyNames[] =
{
	{0,	u"Easy",		u"Table Images/Options/Difficulty/Easy"		},	//	green
	{1,	u"Medium",		u"Table Images/Options/Difficulty/Medium"	},	//	yellow
	{2,	u"Hard",		u"Table Images/Options/Difficulty/Hard"		},	//	red
	{3,	u"Extra Hard",	u"Table Images/Options/Difficulty/Extra Hard"},	//	black
};

typedef struct
{
	ViewType	itsViewType;
	Char16		*itsName,
				*itsImageName;
} ViewModeName;
ViewModeName
	gViewModeNames2D[] =
	{
		{ViewBasicLarge,	u"Basic (2D)",		u"Table Images/Options/View/Basic"		},
		{ViewRepeating,		u"Repeating (2D)",	u"Table Images/Options/View/Repeating"	}
	},
	gViewModeNames3D[] =
	{
		{ViewBasicLarge,	u"Basic (3D)",		u"Table Images/Options/View/Basic"		},
		{ViewRepeating,		u"Repeating (3D)",	u"Table Images/Options/View/Repeating"	}
	};


//	Privately-declared properties and methods
@interface TorusGamesOptionsChoiceController()
- (void)soundEffectsToggled:(id)sender;
@end


@implementation TorusGamesOptionsChoiceController
{
	id<TorusGamesOptionsChoiceDelegate> __weak	itsDelegate;
	bool										itsHumanVsComputerChoice;
	TopologyType								itsTopologyChoice;
	unsigned int								itsDifficultyChoice;
	ViewType									itsViewTypeChoice;
	bool										itsSoundEffectsChoice;
	GameType									itsGame;
	
	TopologyName								*itsTopologyNames;
	unsigned int								itsNumTopologyNames;
	ViewModeName								*itsViewModeNames;
	unsigned int								itsNumViewModeNames;
	
	bool										itsViewChoiceIsIncluded;

	UIBarButtonItem								*itsCancelButton;
}


- (id)	initWithDelegate:	(id<TorusGamesOptionsChoiceDelegate>)aDelegate 
		humanVsComputer:	(bool)aHumanVsComputer 
		topology:			(TopologyType)aTopology
		difficultyLevel:	(unsigned int)aDifficultyLevel
		viewType:			(ViewType)aViewType
		soundEffects:		(bool)aSoundEffects
		duringGame:			(GameType)aGame
{
	self = [super initWithStyle:UITableViewStyleGrouped];
	if (self != nil)
	{
		itsDelegate = aDelegate;

		itsHumanVsComputerChoice	= aHumanVsComputer;
		itsTopologyChoice			= aTopology;
		itsDifficultyChoice			= aDifficultyLevel;
		itsViewTypeChoice			= aViewType;
		itsSoundEffectsChoice		= aSoundEffects;
		
		itsGame = aGame;
		
		if (GameIs3D(itsGame))
		{
			itsTopologyNames	= gTopology3DNames;
			itsNumTopologyNames	= BUFFER_LENGTH(gTopology3DNames);
			
			itsViewModeNames	= gViewModeNames3D;
			itsNumViewModeNames	= BUFFER_LENGTH(gViewModeNames3D);
		}
		else
		{
			itsTopologyNames	= gTopology2DNames;
			itsNumTopologyNames	= BUFFER_LENGTH(gTopology2DNames);
			
			itsViewModeNames	= gViewModeNames2D;
			itsNumViewModeNames	= BUFFER_LENGTH(gViewModeNames2D);
		};
		
		itsViewChoiceIsIncluded = ([[UIDevice currentDevice] userInterfaceIdiom]
									== UIUserInterfaceIdiomPad);

		[[self navigationItem] setTitle:GetLocalizedTextAsNSString(u"Options")];

		//	To make it easy to try Crossword and Word Search puzzles
		//	in different languages, Torus Games lets the player change
		//	languages on the fly.  So let's provide an explicit localization
		//	for the "Cancel" button, in lieu of the built-in
		//
		//		initWithBarButtonSystemItem:UIBarButtonSystemItemCancel
		//
		itsCancelButton = [[UIBarButtonItem alloc]
			initWithTitle:	GetLocalizedTextAsNSString(u"Cancel")
			style:			UIBarButtonItemStylePlain
			target:			aDelegate
			action:			@selector(userDidCancelOptionsChoice)];
	}
	return self;
}

- (BOOL)prefersStatusBarHidden
{
	return [itsDelegate prefersStatusBarHidden];
}

- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];
	
	//	We must call -layoutIfNeeded explicitly, otherwise -contentSize returns CGSizeZero.
	[[self tableView] layoutIfNeeded];
	[self setPreferredContentSize:(CGSize){	PREFERRED_POPOVER_WIDTH,
											[[self tableView] contentSize].height}];

	[self adaptNavBarForHorizontalSize:
		[[RootViewController(self) traitCollection] horizontalSizeClass]];
}


#pragma mark -
#pragma mark GeometryGamesPopover

- (void)adaptNavBarForHorizontalSize:(UIUserInterfaceSizeClass)aHorizontalSizeClass
{
	[[self navigationItem]
		setRightBarButtonItem:	(aHorizontalSizeClass == UIUserInterfaceSizeClassCompact ?
									itsCancelButton : nil)
		animated:				NO];
}


#pragma mark -
#pragma mark UITableViewDataSource

- (NSInteger)numberOfSectionsInTableView:(UITableView *)aTableView
{
#ifdef SCIENCE_CENTER_VERSION
	return 3;	//	suppress the View choice and the Sound Effects switch
#else
	if (itsViewChoiceIsIncluded)
		return 5;	//	include View choice
	else
		return 4;	//	exclude View choice
#endif
}

- (NSString *)tableView:(UITableView *)aTableView titleForHeaderInSection:(NSInteger)aSection
{
#ifdef SCIENCE_CENTER_VERSION
	switch (aSection)
	{
		case 0:		return GetLocalizedTextAsNSString(u"Players"	);
		case 1:		return GetLocalizedTextAsNSString(u"Topology"	);
		case 2:		return GetLocalizedTextAsNSString(u"Difficulty"	);
		default:	return @"internal error";
	}
#else
	if (itsViewChoiceIsIncluded)
	{
		switch (aSection)
		{
			case 0:		return GetLocalizedTextAsNSString(u"Players"	);
			case 1:		return GetLocalizedTextAsNSString(u"Topology"	);
			case 2:		return GetLocalizedTextAsNSString(u"Difficulty"	);
			case 3:		return GetLocalizedTextAsNSString(u"View"		);
			case 4:		return nil;	//	no section title for sound effects
			default:	return @"internal error";
		}
	}
	else
	{
		switch (aSection)
		{
			case 0:		return GetLocalizedTextAsNSString(u"Players"	);
			case 1:		return GetLocalizedTextAsNSString(u"Topology"	);
			case 2:		return GetLocalizedTextAsNSString(u"Difficulty"	);
			case 3:		return nil;	//	no section title for sound effects
			default:	return @"internal error";
		}
	}
#endif
}

- (NSInteger)tableView:(UITableView *)aTableView numberOfRowsInSection:(NSInteger)aSection
{
#ifdef SCIENCE_CENTER_VERSION
	switch (aSection)
	{
		case 0:		return BUFFER_LENGTH(gPlayModeNames);
		case 1:		return itsNumTopologyNames;
		case 2:		return BUFFER_LENGTH(gDifficultyNames);
		default:	return 0;
	}
#else
	if (itsViewChoiceIsIncluded)
	{
		switch (aSection)
		{
			case 0:		return BUFFER_LENGTH(gPlayModeNames);
			case 1:		return itsNumTopologyNames;
			case 2:		return BUFFER_LENGTH(gDifficultyNames);
			case 3:		return itsNumViewModeNames;
			case 4:		return 1;	//	1 row in sound effects section
			default:	return 0;
		}
	}
	else
	{
		switch (aSection)
		{
			case 0:		return BUFFER_LENGTH(gPlayModeNames);
			case 1:		return itsNumTopologyNames;
			case 2:		return BUFFER_LENGTH(gDifficultyNames);
			case 3:		return 1;	//	1 row in sound effects section
			default:	return 0;
		}
	}
#endif
}

- (UITableViewCell *)tableView:(UITableView *)aTableView cellForRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	UITableViewCell	*theCell;
	UILabel			*theLabel;
	NSUInteger		theSection,
					theRow;
	const Char16	*theTitle,
					*theImageName;
	bool			theCellIsSelectable,
					theCheckmark;
	UISwitch		*theSwitch;

	theCell		= [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:nil];
	theLabel	= [theCell textLabel];

	theSection	= [anIndexPath section];
	theRow		= [anIndexPath row];

	if (itsViewChoiceIsIncluded)
	{
		//	Interpret the section number exactly as it is.
	}
	else
	{
		//	Adjust the section number to what it would have been
		//	if the ViewType choice had been included as section 3.
		if (theSection >= 3)
			theSection++;
	}

	switch (theSection)
	{
		case 0:
			theTitle		= GetLocalizedText(gPlayModeNames[theRow].itsName);
			theImageName	= gPlayModeNames[theRow].itsImageName;
			if (itsGame == GameNone
			 || itsGame == Game2DTicTacToe
			 || itsGame == Game2DGomoku
			 || itsGame == Game2DChess
			 || itsGame == Game2DPool
			 || itsGame == Game3DTicTacToe)
			{
				theCellIsSelectable	= true;
				theCheckmark		= (gPlayModeNames[theRow].itsHumanVsComputer == itsHumanVsComputerChoice);
			}
			else
			{
				theCellIsSelectable	= false;
				theCheckmark		= false;
			}
			theSwitch		= nil;
			break;

		case 1:
			theTitle			= GetLocalizedText(itsTopologyNames[theRow].itsName);
			theImageName		= itsTopologyNames[theRow].itsImageName;
			theCellIsSelectable	= true;
			theCheckmark		= (itsTopologyNames[theRow].itsTopology == itsTopologyChoice);
			theSwitch			= nil;
			break;

		case 2:
			theTitle		= GetLocalizedText(gDifficultyNames[theRow].itsName);
			theImageName	= gDifficultyNames[theRow].itsImageName;
			if ((itsGame == Game2DGomoku && itsHumanVsComputerChoice)
			 ||  itsGame == Game2DMaze
			 ||  itsGame == Game2DJigsaw
			 || (itsGame == Game2DChess  && itsHumanVsComputerChoice)
			 || (itsGame == Game2DPool   && itsHumanVsComputerChoice)
			 ||  itsGame == Game2DApples
			 ||  itsGame == Game3DMaze)
			{
				theCellIsSelectable	= true;
				theCheckmark		= (gDifficultyNames[theRow].itsDifficultyLevel == itsDifficultyChoice);
			}
			else
			{
				theCellIsSelectable	= false;
				theCheckmark		= false;
			}
			theSwitch		= nil;
			break;

		case 3:
			theTitle			= GetLocalizedText(itsViewModeNames[theRow].itsName);
			theImageName		= itsViewModeNames[theRow].itsImageName;
			theCellIsSelectable	= true;
			theCheckmark		= (itsViewModeNames[theRow].itsViewType == itsViewTypeChoice);
			theSwitch			= nil;
			break;

		case 4:
			theTitle			= GetLocalizedText(u"Sound Effects");
			theImageName		= u"Table Images/Options/Sound Effects";
			theCellIsSelectable	= false;
			theCheckmark		= false;	//	accessoryType is ignored when accessoryView ≠ nil
			theSwitch			= [[UISwitch alloc] initWithFrame:CGRectZero];
			[theSwitch setOn:itsSoundEffectsChoice];
			[theSwitch addTarget:self action:@selector(soundEffectsToggled:) forControlEvents:UIControlEventValueChanged];
			break;

		default:	//	should never occur
			theTitle			= u"?";
			theImageName		= u"?";
			theCellIsSelectable	= false;
			theCheckmark		= false;
			theSwitch			= nil;
			break;
	}

	[theLabel setText:GetNSStringFromZeroTerminatedString(theTitle)];

	[[theCell imageView] setImage:[UIImage imageNamed:
		GetNSStringFromZeroTerminatedString(theImageName)]];

	if (theSwitch == nil
	 && ! theCellIsSelectable )
	{
		[theLabel setTextColor:[UIColor grayColor]];
	}
	//
	//	Note:  It's tempting to replace the following line with
	//
	//		[theCell setUserInteractionEnabled:(BOOL)theCellIsSelectable]
	//
	//	which would avoid having to test the selectionStyle
	//	in -tableView:willSelectRowAtIndexPath:, but catch is that
	//	disabling user interaction in the cell also disables
	//	user access to theSwitch (if present).  There seems to be
	//	no clean way to make the cell nonselectable without
	//	blocking access to the switch that it contains.
	//
	if ( ! theCellIsSelectable )
		[theCell setSelectionStyle:UITableViewCellSelectionStyleNone];

	[theCell setAccessoryType:(theCheckmark ?
				UITableViewCellAccessoryCheckmark : 
				UITableViewCellAccessoryNone)];

	[theCell setAccessoryView:theSwitch];

	return theCell;
}

- (void)soundEffectsToggled:(id)sender
{
	if ([sender isKindOfClass:[UISwitch class]])	//	should never fail
	{
		itsSoundEffectsChoice = [((UISwitch *)sender) isOn];
		[itsDelegate userDidChooseOptionSoundEffects:itsSoundEffectsChoice];
	}
}


#pragma mark -
#pragma mark UITableViewDelegate

- (NSIndexPath *)tableView:(UITableView *)aTableView willSelectRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	NSUInteger	theSection,
				theRow;
	
	//	Ignore a tap on a non-selectable cell.
	if ([[aTableView cellForRowAtIndexPath:anIndexPath] selectionStyle] == UITableViewCellSelectionStyleNone)
		return nil;

	theSection	= [anIndexPath section];
	theRow		= [anIndexPath row];

	if (itsViewChoiceIsIncluded)
	{
		//	Interpret the section number exactly as it is.
	}
	else
	{
		//	Adjust the section number to what it would have been
		//	if the ViewType choice had been included as section 3.
		if (theSection >= 3)
			theSection++;
	}

	switch (theSection)
	{
		case 0:
			itsHumanVsComputerChoice	= gPlayModeNames[theRow].itsHumanVsComputer;
			break;

		case 1:
			itsTopologyChoice			= itsTopologyNames[theRow].itsTopology;
			break;

		case 2:
			itsDifficultyChoice			= gDifficultyNames[theRow].itsDifficultyLevel;
			break;

		case 3:
			itsViewTypeChoice			= itsViewModeNames[theRow].itsViewType;
			break;

		case 4:
			//	Should never occur.
			//	The Sound Effects button is non-selectable.
			//	The user must slide the toggle switch
			//	to turn Sound Effects on or off.
			return nil;
	}	

	//	Reload the table so the checkmark will appear by the newly selected option.
	//	The user will see it change briefly as the table animates away.
	[[self tableView] reloadData];

	//	Report the new option to the caller,
	//	who will dismiss this TorusGamesOptionsChoiceController.
	switch (theSection)
	{
		case 0:
			[itsDelegate userDidChooseOptionHumanVsComputer:itsHumanVsComputerChoice];
			break;

		case 1:
			[itsDelegate userDidChooseOptionTopology:itsTopologyChoice];
			break;

		case 2:
			[itsDelegate userDidChooseOptionDifficultyLevel:itsDifficultyChoice];
			break;

		case 3:
			[itsDelegate userDidChooseOptionViewType:itsViewTypeChoice];
			break;

		case 4:
			//	Should never occur.
			//	See explanation above.
			break;
	}	

	return nil;
}

@end
